import argparse, os, json, numpy as np, pandas as pd
from .config import load_study_config
from .parse_sparc import read_mrt_table, split_mass_models_by_galaxy, extract_columns, read_sizes_from_table1
from .rar_core import compute_g, build_vbar, select_outer_window, compute_gstar_AR, compute_gstar_S

def main():
    ap = argparse.ArgumentParser(description="Run T2 (RAR with AR scaling).")
    ap.add_argument("--mass-models", required=True)
    ap.add_argument("--sample-table", default="")
    ap.add_argument("--sizes-csv", default="")
    ap.add_argument("--config", default="config/study.yaml")
    args = ap.parse_args()

    cfg = load_study_config(args.config)
    mm = read_mrt_table(args.mass_models)
    groups = split_mass_models_by_galaxy(mm)

    size_map={}
    if cfg.size_rule in ["RG=2.2*Rd","RG=Reff"] and args.sample_table:
        size_map = read_sizes_from_table1(args.sample_table, cfg.size_rule)
    elif cfg.size_rule=="RG=from_csv":
        import pandas as pd, os
        path = args.sizes_csv or cfg.sizes_csv
        if os.path.exists(path):
            df = pd.read_csv(path); size_map = dict(zip(df['gal_id'].astype(str), df['R_G_kpc'].astype(float)))

    rows_u=[]; rows_s=[]; windows={}
    for gal_id, gdf in groups.items():
        try:
            R, Vobs, comps = extract_columns(gdf)
        except Exception:
            continue

        used = np.ones_like(R.values, dtype=bool)
        if cfg.use_outer_window_only:
            i0,i1,ok = select_outer_window(R.values, Vobs.values, cfg.dvdr_max_kms_per_kpc, cfg.min_window_points, cfg.min_radius_kpc, cfg.min_window_span_kpc)
            if ok:
                used[:] = False; used[i0:i1+1] = True
                windows[gal_id]={"rmin_kpc": float(R.values[i0]), "rmax_kpc": float(R.values[i1])}
            else:
                used[:] = False
                windows[gal_id]={"rmin_kpc": None, "rmax_kpc": None}

        Vbar = None
        try:
            Vbar = build_vbar(comps.get('Vgas'), comps.get('Vdisk'), comps.get('Vbulge'))
        except Exception:
            pass
        if Vbar is None: 
            continue

        g_obs = compute_g(R.values, Vobs.values)
        g_bar = compute_g(R.values, Vbar)

        RG = size_map.get(gal_id, np.nan)
        if RG==RG:
            if cfg.mode.upper()=="AR":
                gstar = compute_gstar_AR(cfg.alphaP, cfg.lambda_G_default, cfg.c_ms, cfg.UGM_phys_m, cfg.R_obs_m, RG)
            else:
                gstar = compute_gstar_S(cfg.S_s2, RG)
        else:
            gstar = np.nan

        for r, gb, go, use in zip(R.values, g_bar, g_obs, used):
            rows_u.append({"gal_id": gal_id, "r_kpc": float(r), "g_bar": float(gb), "g_obs": float(go), "used_in_window": bool(use)})
            if gstar==gstar:
                rows_s.append({"gal_id": gal_id, "r_kpc": float(r), "g_star": float(gstar), "gbar_scaled": float(gb/gstar), "gobs_scaled": float(go/gstar), "used_in_window": bool(use)})

    os.makedirs("outputs", exist_ok=True)
    pd.DataFrame(rows_u).to_csv("outputs/rar_unscaled.csv", index=False)
    pd.DataFrame(rows_s).to_csv("outputs/rar_ARscaled.csv", index=False)
    with open("outputs/windows.json","w") as f: json.dump(windows, f, indent=2)

if __name__=="__main__":
    main()
